/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: TransportLayer.cpp 6860 2015-04-16 13:58:33Z Robert $
   Last Modification:
    $Author: Robert $
    $Date: 2015-04-16 15:58:33 +0200 (Do, 16 Apr 2015) $
    $Revision: 6860 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
     Implementation of the "Transport Layer" Class, which handles the hilscher transport
     on a single pyhsical interface


   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     12        16.04.15    LC       added 64bit support: abs() needs correct parameter data type
     11        31.03.11    SS       Send queue not cleared at deinit causes memory leak
     10        15.12.10    RM       KeepAlive is now deactivated when compiled in DEBUG mode
     9         25.11.10    MT       - Bugfix: Invalid incoming packets with valid header 
                                              may crash the dll (new throws an exception)
                                    - Bugfix: Validate size of send queue before deleting entries
     8         05.10.10    SS       - Bugfix: xChannel/SysdevicePutPacket times out 
                                      due to missing interface specific transmit 
                                      timeout (rcX packet transfer)
                                    - Unused variables removed
     7         30.06.10    SS       - Interface specific transmit timeout is now used 
                                      for the packet transfer.
                                    - Bugfix: Checksum calculation omitted for put
                                      packet (rcX Packet transfer) 
     6         02.06.10    SS       - Bugfix: Checksum in transport frame omitted .
                                      Due to compatibility reasons the checksum is still 
                                      not used in conjunction with to the old transport 
                                      protocol implementation.
                                    - Bugfix: Acknowledge not send as a result of a call 
                                      to xSysdevicePutPacket() and xChannelPutPacket()
                                      (only rcX Packet Transfer). Workaround for old 
                                      transport protocol implementation removed.
     5         29.04.10    SS       - Accelerate packet transfer for old transport protocol
                                      implementation (sleep 100ms after every 5th send packet)
                                    - Bugfix: Sleep required in old transport protocol 
                                      implementation may not performed if packets send 
                                      simultaneously by multiple threads
     4         03.12.09    RM       Review
     3         06.11.09    PL       Structure change of data handling
     2         03.03.09    MT       Structural change to allow placing physical drivers
                                    in a dll
     
     1         25.09.08    PL       initial version
 
**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file TransportLayer.cpp
/// Implementation of the "Transport Layer" Class, which handles the hilscher transport
/// on a single pyhsical interface
/////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "APIInterface.h"
#include "DeviceHandler.h"
#include "PhysicalInterface.h"
#include "TransportLayer.h"
#include "cifXErrors.h"
#include "Marshaller.h"
#include "rcXPacket.h"
#include "HelperFunctions.h"

uint32_t g_ulTxSemaphoreFails = 0;

///////////////////////////////////////////////////////////////////////////////////////////
/// Create a CIFX error from a HIL transport state
///   \param  bState  State from the HIL transport header
///   \return CIFX ERROR Corresponding CIFX error
///////////////////////////////////////////////////////////////////////////////////////////
static int32_t RetFromState(BYTE bState)
{
  int32_t lRet;

  switch(bState)
  {
  case HIL_TRANSPORT_STATE_OK:
    lRet = CIFX_NO_ERROR;
    break;

  case HIL_TSTATE_CHECKSUM_ERROR:
    lRet = CIFX_TRANSPORT_CHECKSUM_ERROR;
    break;

  case HIL_TSTATE_LENGTH_INCOMPLETE:
    lRet = CIFX_TRANSPORT_LENGTH_INCOMPLETE;
    break;

  case HIL_TSTATE_DATA_TYPE_UNKNOWN:
    lRet = CIFX_TRANSPORT_DATA_TYPE_UNKOWN;
    break;

  case HIL_TSTATE_DEVICE_UNKNOWN:
    lRet = CIFX_TRANSPORT_DEVICE_UNKNOWN;
    break;

  case HIL_TSTATE_CHANNEL_UNKNOWN:
    lRet = CIFX_TRANSPORT_CHANNEL_UNKNOWN;
    break;

  case HIL_TSTATE_SEQUENCE_ERROR:
    lRet = CIFX_TRANSPORT_SEQUENCE;
    break;

  case HIL_TSTATE_BUFFEROVERFLOW_ERROR:
    lRet = CIFX_TRANSPORT_BUFFEROVERFLOW;
    break;

  case HIL_TSTATE_KEEP_ALIVE_ERROR:
    lRet = CIFX_TRANSPORT_KEEPALIVE;
    break;

  case HIL_TSTATE_RESOURCE_ERROR:
    lRet = CIFX_TRANSPORT_RESOURCE;
    break;

  default:
    lRet = CIFX_TRANSPORT_ERROR_UNKNOWN;
    break;
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Constructor of the CTransportLayer class
///   \param pcInterface     Handle of the connected interface
///   \param pcDeviceHandler Handle of the connected device handler
///////////////////////////////////////////////////////////////////////////////////////////
CTransportLayer::CTransportLayer(CPhysicalInterface* pcInterface, CDeviceHandler* pcDeviceHandler)
  : m_pcInterface(pcInterface)
  , m_pcDeviceHandler(pcDeviceHandler)
  , m_lRefCount(0)
  , m_bSequenceNr(0)
  , m_usTransactionNr(0)
  , m_hTransportThread(NULL)
  , m_hTransportThreadStop(NULL)
  , m_hRxSemaphore(NULL)
  , m_hTxSemaphore(NULL)
  , m_hKeepAliveThread(NULL)
  , m_hKeepAliveStop(NULL)
  , m_hKeepAliveEvent(NULL)
  , m_eKeepAliveState(eKEEP_ALIVE_UNSUPPORTED)
  , m_ulKeepAliveIdent(0)
  , m_ulLastKeepAlive(0)
  , m_ulKeepAliveTimeout(HIL_TRANSPORT_KEEP_ALIVE_CLIENT_TIMEOUT)
{
  InitializeCriticalSection(&m_tcsHeader);
  InitializeCriticalSection(&m_tcsRxQueue);
  InitializeCriticalSection(&m_tcsRxLock);
  InitializeCriticalSection(&m_tcsTransferLock);
  InitializeCriticalSection(&m_tcsTxQueue);
  InitializeCriticalSection(&m_tcsInService);
  InitializeCriticalSection(&m_tcsActiveServices);
  
  m_tServerInfo.ulBufferSize       = sizeof(CIFX_PACKET);
  m_tServerInfo.ulParallelServices = 1;

  memset(&m_tRxCurrent, 0, sizeof(m_tRxCurrent));

  // Default "Transport Packet"
  memset(&m_tACKPacket , 0, sizeof(m_tACKPacket ));
  m_tACKPacket.usDataType = HIL_TRANSPORT_TYPE_ACKNOWLEDGE;
  m_tACKPacket.ulCookie   = HIL_TRANSPORT_COOKIE;

  // Default "Keep Alive Packet"
  memset(&m_tKeepAlivePacket, 0, sizeof(m_tKeepAlivePacket));
  m_tKeepAlivePacket.tHeader.ulLength   = sizeof(HIL_TRANSPORT_KEEPALIVE_DATA_T);
  m_tKeepAlivePacket.tHeader.usDataType = HIL_TRANSPORT_TYPE_KEEP_ALIVE;
  m_tKeepAlivePacket.tHeader.ulCookie   = HIL_TRANSPORT_COOKIE;
  m_tKeepAlivePacket.pbData             = (BYTE*)&m_ulKeepAliveIdent;

  m_ulSendTimeout = m_pcInterface->GetSendTimeout();
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Denstructor of the CTransportLayer class
///////////////////////////////////////////////////////////////////////////////////////////
CTransportLayer::~CTransportLayer()
{
  DeInit();

  DeleteCriticalSection(&m_tcsHeader);
  DeleteCriticalSection(&m_tcsRxQueue);
  DeleteCriticalSection(&m_tcsRxLock);
  DeleteCriticalSection(&m_tcsTransferLock);
  DeleteCriticalSection(&m_tcsTxQueue);
  DeleteCriticalSection(&m_tcsInService);
  DeleteCriticalSection(&m_tcsActiveServices);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Get the interface name
///   \param szName String buffer
///////////////////////////////////////////////////////////////////////////////////////////
void  CTransportLayer::GetInterfaceName(std::string& szName)
{ 
  m_pcInterface->GetInterfaceName(szName); 
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Calculate CRC16 checksum for HIL_TRANSPORT_HEADER
///  \param  pbData    Data pointer
///  \param  ulDataLen Length of data
///  \return CRC16 Checksum
///////////////////////////////////////////////////////////////////////////////////////////
uint16_t CTransportLayer::CalcCrc16(uint8_t* pbData, uint32_t ulDataLen)
{
	/* CRC16-CCIT Polynom */
	uint16_t usCrcValue = 0xFFFF;

	for(uint32_t ulIdx = 0; ulIdx < ulDataLen; ulIdx++)
	{
		usCrcValue  = (usCrcValue >> 8) | (usCrcValue << 8);
		usCrcValue ^= pbData[ulIdx];
		usCrcValue ^= (usCrcValue & 0xFF) >> 4;
		usCrcValue ^= (usCrcValue << 8) << 4;
		usCrcValue ^= ((usCrcValue & 0xFF) << 4) << 1;
	}

  return ~usCrcValue;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// This is a checksum used on first release netIC Firmware version. It is NO Crc checksum,
/// but we need it to be "old" compatible
///  \param  pbData    Data pointer
///  \param  ulDataLen Length of data
///  \return netIC Checksum
///////////////////////////////////////////////////////////////////////////////////////////
uint16_t CTransportLayer::CalcCrcNetIC(uint8_t* pbData, uint32_t ulDataLen)
{
  uint16_t usCrcValue = 0xFFFF;

  #define lo8(_x_) (BYTE)(_x_ & 0xFF)
  #define hi8(_x_) (BYTE)((_x_ >> 8) & 0xFF)

	for(uint32_t ulIdx = 0; ulIdx < ulDataLen; ulIdx++)
	{
    BYTE data = pbData[ulIdx];

	  data ^= lo8(usCrcValue);
	  data ^= data << 4;
    
    usCrcValue = ((((WORD)data << 8) | hi8(usCrcValue)) ^ (BYTE)(data >> 4) ^ ((WORD)data << 3));
  }

  return usCrcValue;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Allocate specific memory for a specific packet 
///   \param  ulDataSize            Size of allocate area  
///   \return HIL_TRANSPORT_PACKET* Pointer to alleacted memory
///////////////////////////////////////////////////////////////////////////////////////////
HIL_TRANSPORT_PACKET* CTransportLayer::AllocateTransportFrame(uint32_t ulDataSize)
{
  // Create transport packet
  HIL_TRANSPORT_PACKET* ptRet = new HIL_TRANSPORT_PACKET;

  _ASSERT(ptRet);

  if(ptRet != NULL)
  {
    // Allocate memory for packet data 
    if(NULL == (ptRet->pbData = new uint8_t[ulDataSize]))
    {
      _ASSERT(false);
      delete ptRet;
      ptRet = NULL;
    } else
    {
      if(NULL != ptRet)
      { 
        // Clean memory 
        memset(&ptRet->tHeader, 0, sizeof(HIL_TRANSPORT_HEADER));
        // Check data size
        if(ulDataSize > 0)
        {
          // Set packet data to zero
          memset(ptRet->pbData, 0, ulDataSize);
        }

      } else
      {
        _ASSERT(false);
      }
    }
  }
  return ptRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Delete specific memory of a specific packet 
///   \param  ptPacket  Pointer of allocate area
///   \return void
///////////////////////////////////////////////////////////////////////////////////////////
void CTransportLayer::FreeTransportFrame(HIL_TRANSPORT_PACKET* ptPacket)
{
  if(NULL != ptPacket)
  {
    delete [] ptPacket->pbData;
    delete ptPacket;
  }
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Generate packet, fill specific packet of the Hilscher Transport Header
///   \param  ptHeader  Pointer of packet header 
///   \return void
///////////////////////////////////////////////////////////////////////////////////////////
void CTransportLayer::GeneratePacketHeader(PHIL_TRANSPORT_HEADER ptHeader)
{
  ptHeader->ulCookie      = HIL_TRANSPORT_COOKIE;

  EnterCriticalSection(&m_tcsHeader);

  ptHeader->bSequenceNr   = ++m_bSequenceNr;
  ptHeader->usTransaction = ++m_usTransactionNr;

  LeaveCriticalSection(&m_tcsHeader);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Initialization of the transport layer object 
///   \return  CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t  CTransportLayer::Init( void)
{
  int32_t lRet = CIFX_NO_ERROR;

  if(InterlockedIncrement(&m_lRefCount) == 1)
  {
    lRet = CIFX_FUNCTION_FAILED;

    m_cvActiveServices.clear();

    // Start transport layer transfer thread
    if( (NULL == (m_hRxSemaphore = ::CreateSemaphore(NULL, 0, MAXLONG, NULL))) ||
        (NULL == (m_hTxSemaphore = ::CreateSemaphore(NULL, 0, MAXLONG, NULL))) ||
        (NULL == (m_hTransportThreadStop = ::CreateEvent(NULL, FALSE, FALSE, NULL))) )
    {
      /* Failed to create semaphore */
      lRet = CIFX_DRV_INIT_ERROR;

    } else if(NULL == (m_hTransportThread = ::CreateThread(NULL,
                                                           0,
                                                           TransportThread,
                                                           this,
                                                           0,
                                                           NULL)))
    {
      /* Failed to create transport thread */
      lRet = CIFX_DRV_INIT_ERROR;
    } else if( (NULL == (m_hKeepAliveEvent = ::CreateEvent(NULL, FALSE, FALSE, NULL))) || 
               (NULL == (m_hKeepAliveStop  = ::CreateEvent(NULL, FALSE, FALSE, NULL))) )
    {
      lRet = CIFX_DRV_INIT_ERROR;
    } else if(NULL == (m_hKeepAliveThread = ::CreateThread(NULL, 0, KeepAliveThread, this, 0, NULL)))
    {
      lRet = CIFX_DRV_INIT_ERROR;
    } else
    {
      lRet = CIFX_NO_ERROR;
    }

    if(CIFX_NO_ERROR != lRet)
      DeInit();

  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// DeInitialize the transport layer object 
///   \return  CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t  CTransportLayer::DeInit( void)
{ 
  int32_t lRet = CIFX_NO_ERROR;

  if(InterlockedDecrement(&m_lRefCount) <= 0)
  {
    /* Stop all pending transfers */
    EnterCriticalSection(&m_tcsTransferLock);
    while(m_cvPendingTransfers.size() > 0)
    {
      TRANSFER_INFO_T* ptTransfer = m_cvPendingTransfers[0];

      ptTransfer->fAborted = true;
      ::SetEvent(ptTransfer->hWaitEvent);

      m_cvPendingTransfers.erase(m_cvPendingTransfers.begin());
    }
    LeaveCriticalSection(&m_tcsTransferLock);

    /* Clear send queue */
    EnterCriticalSection(&m_tcsTxQueue);
    while(m_cTxQueue.size() > 0)
    {
      FreeTransportFrame(m_cTxQueue.front());
      m_cTxQueue.pop_front();
    }
    LeaveCriticalSection(&m_tcsTxQueue);

    /* Stop transport thread */
    if(m_hKeepAliveThread != NULL)                                              
    {
      ::SetEvent(m_hKeepAliveStop);

      if(WaitForSingleObject( m_hKeepAliveThread, 1000) != WAIT_OBJECT_0)
        ::TerminateThread( m_hKeepAliveThread, MAXDWORD);

      CloseHandle(m_hKeepAliveThread);
      m_hKeepAliveThread = NULL;
    }

    if(NULL != m_hKeepAliveStop)
    {
      ::CloseHandle(m_hKeepAliveStop);
      m_hKeepAliveStop = NULL;
    }

    if(NULL != m_hKeepAliveEvent)
    {
      ::CloseHandle(m_hKeepAliveEvent);
      m_hKeepAliveEvent = NULL;
    }

    // Stop transport thread 
    if(m_hTransportThread != NULL)                                              
    {
      ::SetEvent(m_hTransportThreadStop);

      if(WaitForSingleObject( m_hTransportThread, m_ulSendTimeout + 1000) != WAIT_OBJECT_0)
        ::TerminateThread( m_hTransportThread, MAXDWORD);

      CloseHandle(m_hTransportThread);
      m_hTransportThread = NULL;
    }

    if(NULL != m_hRxSemaphore)
    {
      ::CloseHandle(m_hRxSemaphore);
      m_hRxSemaphore = NULL;
    }

    if(NULL != m_hTxSemaphore)
    {
      ::CloseHandle(m_hTxSemaphore);
      m_hTxSemaphore = NULL;
    }

    if(NULL != m_hTransportThreadStop)
    {
      ::CloseHandle(m_hTransportThreadStop);
      m_hTransportThreadStop = NULL;
    }

    m_lRefCount = 0;
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Transport layer thread member function
///   \param  pvParam User information
///   \return 0 always
///////////////////////////////////////////////////////////////////////////////////////////
DWORD CTransportLayer::TransportThread(void* pvParam)
{
  CTransportLayer* pcLayer = reinterpret_cast<CTransportLayer*>(pvParam);

  pcLayer->TransportThreadFunc();

  return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Transport layer send/receive worker thread
///////////////////////////////////////////////////////////////////////////////////////////
void CTransportLayer::TransportThreadFunc(void)
{
  bool fRunning = true;
  
  // Set Events
  HANDLE  ahWaitEvents[] = { m_hRxSemaphore,
                             m_hTxSemaphore,
                             m_hTransportThreadStop};

  while(fRunning)
  {
    DWORD dwWaitRes = ::WaitForMultipleObjects( sizeof(ahWaitEvents) / sizeof(ahWaitEvents[0]),
                                                ahWaitEvents,
                                                FALSE,
                                                m_ulKeepAliveTimeout);
    switch(dwWaitRes)
    {
      case WAIT_OBJECT_0:     // Receive packet
      {
        PHIL_TRANSPORT_PACKET ptPacket;

        EnterCriticalSection(&m_tcsRxQueue);

        ptPacket = m_cRxQueue.front();
        m_cRxQueue.pop_front();

        LeaveCriticalSection(&m_tcsRxQueue);

        bool fDeletePacket = true;

        switch(ptPacket->tHeader.usDataType)
        {
          case HIL_TRANSPORT_TYPE_ACKNOWLEDGE:
          {
            EnterCriticalSection(&m_tcsInService);

            HIL_INSERV_MAP::iterator iter = m_cmInService.find(ptPacket->tHeader.bSequenceNr);

            if(iter != m_cmInService.end())
            {
              if(NULL != iter->second.pbAckState)
                *iter->second.pbAckState = ptPacket->tHeader.bState;

              if(NULL != iter->second.hWaitAckEvent)
                ::SetEvent(iter->second.hWaitAckEvent);

              m_cmInService.erase(iter);
            }

            LeaveCriticalSection(&m_tcsInService);
          }
          break;

          /* HIL_TRANSPORT_TYPE_KEEP_ALIVE:
              NOTE: Keep alive is handled synchronously in timeout handle and will never arrive here,
                    but must be handled in default case */
          case HIL_TRANSPORT_TYPE_QUERYDEVICE:
          case HIL_TRANSPORT_TYPE_QUERYSERVER:
            RemoveParallelService();

            if (m_cTxQueue.size() > 0)     
              AddParallelService(CIFX_TO_1ST_PACKET);

            SendAcknowledge(&ptPacket->tHeader, HIL_TRANSPORT_STATE_OK);
            
            /* If the transfer was active, we must not delete the packet, as
              it is needed by a caller to TransferPacket */
            if(HandlePendingTransfers(ptPacket))
            {
              fDeletePacket = false;
              /* Free the transport packet struct is usually done by calling the FreeTransportFrame() routine, which
                 frees the struct and its data area. As we still need the data area, we only delete the transport packet 
                 struct here. The data area of the packet struct is handed back to the function which initiated the 
                 transfer, so we must not delete this area */
              delete ptPacket;
            }
          break;

          default:
          {
            RemoveParallelService();
            if (m_cTxQueue.size() > 0)     
              AddParallelService(CIFX_TO_1ST_PACKET);

            SendAcknowledge(&ptPacket->tHeader, HIL_TRANSPORT_STATE_OK);

            if(HandlePendingTransfers(ptPacket))
            {
              fDeletePacket = false;
              /* Free the transport packet struct is usually done by calling the FreeTransportFrame() routine, which
                 frees the struct and its data area. As we still need the data area, we only delete the transport packet 
                 struct here. The data area of the packet struct is handed back to the function which initiated the 
                 transfer, so we must not delete this area */
              delete ptPacket;

            } else
            {
              /* Distribute packet to registered data layers */
              std::string szInterface;
              m_pcInterface->GetInterfaceName(szInterface);
              
              CEndpoint* pcEndpoint = m_pcDeviceHandler->GetEndpoint(szInterface, ptPacket->tHeader.bDevice);
                
              if(NULL != pcEndpoint)
              {
                std::map<uint16_t, CDataLayer*>::iterator iter = pcEndpoint->m_cmDataLayers.find(ptPacket->tHeader.usDataType);

                if(iter != pcEndpoint->m_cmDataLayers.end())
                {
                  CDataLayer* pcLayer = iter->second;

                  fDeletePacket = pcLayer->ReceiveData(ptPacket);
                }
              }
            }
          }
          break;
        } // end switch

        if(fDeletePacket)
          FreeTransportFrame(ptPacket);
      }
      break;

      case WAIT_OBJECT_0 + 1: // Send packet
      {
        uint32_t         ulDataLength = 0;
        HIL_TRANSPORT_PACKET* ptPacket     = NULL;
        BYTE*                 pbData       = NULL;

        // Lock Send queue
        EnterCriticalSection(&m_tcsTxQueue);

        if(0 != m_cTxQueue.size())
        {
          ptPacket = m_cTxQueue.front();
          m_cTxQueue.pop_front();

          LeaveCriticalSection(&m_tcsTxQueue);

          ulDataLength = ptPacket->tHeader.ulLength + sizeof(ptPacket->tHeader);
            
          pbData = new BYTE[ulDataLength];

          memcpy(pbData, &ptPacket->tHeader, sizeof(ptPacket->tHeader));
          memcpy(pbData + sizeof(ptPacket->tHeader), ptPacket->pbData, ptPacket->tHeader.ulLength);

          int32_t lTemp;

          lTemp = m_pcInterface->Send(pbData, ulDataLength);
          if(CIFX_NO_ERROR != lTemp)
          {
            /* Tell Devicehandler we want to reconnect */
            m_pcDeviceHandler->ScheduleInterfaceMonitor(m_pcInterface->GetInterfaceName(), CDeviceHandler::eRECONNECT);
          }

          // Clear allocated memory of the pbData
          delete[] pbData;
          FreeTransportFrame(ptPacket);
        } else
        {
          LeaveCriticalSection(&m_tcsTxQueue);
          g_ulTxSemaphoreFails++;
        }
      }
      break;

      case WAIT_OBJECT_0 + 2: // stop
        fRunning = false;
      break;

      case WAIT_TIMEOUT:
      {
        /* Handle timeout */
        EnterCriticalSection(&m_tcsRxLock);

        // We need to handle keep alive and receive abort 
        DWORD dwCurTime  = GetTickCount();
        DWORD dwTimeDiff = abs((long)dwCurTime - (long)m_ulLastRxTime);

        // Check if we are currently receiving 
        if(m_tRxCurrent.eRxState != eWAIT_FOR_COOKIE)
        {
          if(dwTimeDiff > ABORT_RECEIVE_TIMEOUT)
          {
            if(m_tRxCurrent.tPacket.pbData != NULL)
            {
              delete [] m_tRxCurrent.tPacket.pbData;
              m_tRxCurrent.tPacket.pbData = NULL;
            }
            /* Prepare for new packet */
            m_tRxCurrent.eRxState                 = eWAIT_FOR_COOKIE;
            m_tRxCurrent.ulDataOffset             = 0;
            m_tRxCurrent.ulHeaderOffset           = 0;
            m_tRxCurrent.tPacket.tHeader.ulCookie = 0;
            m_tRxCurrent.tPacket.pbData           = NULL;
          }
        }

        LeaveCriticalSection(&m_tcsRxLock);

        if(m_eKeepAliveState == eKEEP_ALIVE_ACTIVE)
        {
          ::SetEvent(m_hKeepAliveEvent);
        }

        HandleParallelServiceTimeout();
      }
      break;

      default:
        _ASSERT(false);
      break;
    } // end switch
  }
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Receive data callback (used by CPhysicalInterface)
///   \param    pbData    Raw binary data indicated by interface
///   \param    ulDataLen Length of binary data
///////////////////////////////////////////////////////////////////////////////////////////
void CTransportLayer::ReceiveData(uint8_t* pbData, uint32_t ulDataLen)
{
  // Reassemble packet 
  uint32_t ulOffset        = 0;
  bool          fDispatchPacket = true;
  bool          fParsing        = true;

  EnterCriticalSection(&m_tcsRxLock);

#ifdef TRACE_RECEIVED_DATA
  char bBuf[4096];
  static uint32_t cnt = 0;
  cnt+=ulDataLen;
  bBuf[0] = '\0';
  for(int i= 0; i<ulDataLen; i++)
  {
    char buf[3];
    sprintf(buf, "%02x", pbData[i]);
    strcat(bBuf, buf);
  }
  OutputDebugString(bBuf);
  TRACE("Bytes Out: 0x%x", cnt);
#endif

  while(fParsing)
  {
    fParsing = (ulOffset < ulDataLen);
    
    switch(m_tRxCurrent.eRxState)
    {
      case eWAIT_FOR_COOKIE:
      {
        // We need to check fParsing, as it might happen that we run through
        //   the state machine an additional time, and will destroy the cookie in
        //   this case. All other cases check ulOffset explicitely 
        if(fParsing)
        {
          m_tRxCurrent.tPacket.tHeader.ulCookie <<= 8;
          m_tRxCurrent.tPacket.tHeader.ulCookie |= pbData[ulOffset];

          ++m_tRxCurrent.ulHeaderOffset;

          if(HIL_TRANSPORT_COOKIE == m_tRxCurrent.tPacket.tHeader.ulCookie)
          {
            m_tRxCurrent.eRxState         = eWAIT_FOR_HEADER;

          } else if(m_tRxCurrent.ulHeaderOffset >= 4)
          {
            m_tRxCurrent.ulHeaderOffset = 3;
          }
          ++ulOffset;
        }
      }
      break;

      case eWAIT_FOR_HEADER:    // Wait that the complet header is receives 
      { 
        uint8_t* pbHeader  = (uint8_t*)&m_tRxCurrent.tPacket.tHeader;
        uint32_t  ulCopyLen = min(ulDataLen - ulOffset, 
                                       sizeof(m_tRxCurrent.tPacket.tHeader) - m_tRxCurrent.ulHeaderOffset);
       
        memcpy(&pbHeader[m_tRxCurrent.ulHeaderOffset], 
               &pbData[ulOffset],
               ulCopyLen);

        m_tRxCurrent.ulHeaderOffset += ulCopyLen;

        if(m_tRxCurrent.ulHeaderOffset >= sizeof(m_tRxCurrent.tPacket.tHeader))
        {
          if(m_tRxCurrent.tPacket.tHeader.ulLength > 0)
          {
            _ASSERT(m_tRxCurrent.tPacket.tHeader.usDataType != HIL_TRANSPORT_TYPE_ACKNOWLEDGE);

            m_tRxCurrent.eRxState = eWAIT_FOR_PACKETCOMPLETE;
            try
            {
              m_tRxCurrent.tPacket.pbData = new uint8_t[m_tRxCurrent.tPacket.tHeader.ulLength];
            } catch (...)
            {
              /* Error allocating enough memory, so we need to discard this packet and wait for next packet */
              m_tRxCurrent.eRxState                 = eWAIT_FOR_COOKIE;
              m_tRxCurrent.ulDataOffset             = 0;
              m_tRxCurrent.ulHeaderOffset           = 0;
              m_tRxCurrent.tPacket.tHeader.ulCookie = 0;
              m_tRxCurrent.tPacket.pbData           = NULL;
            }
          } else
          {
            /* Send positive Acknowledge */
            m_tRxCurrent.eRxState = ePACKET_COMPLETE;
          }
        }

        ulOffset += ulCopyLen;
      }
      break;

      case eWAIT_FOR_PACKETCOMPLETE:     // Wait, that the received packet is complet, comparing given length in header
      {
        uint32_t ulCopyLen = min(  ulDataLen - ulOffset, 
                                         m_tRxCurrent.tPacket.tHeader.ulLength - m_tRxCurrent.ulDataOffset);

        memcpy(m_tRxCurrent.tPacket.pbData + m_tRxCurrent.ulDataOffset, 
               &pbData[ulOffset], 
               ulCopyLen);
        
        m_tRxCurrent.ulDataOffset += ulCopyLen;
        ulOffset                  += ulCopyLen;

        if(m_tRxCurrent.ulDataOffset >= m_tRxCurrent.tPacket.tHeader.ulLength)
        {
          // Verify Checksum 
          if(m_tRxCurrent.tPacket.tHeader.usChecksum != 0)
          {
            // First we check against the correct CRC16 checksum. If it does not match it might
            //   be an old netIC Firmware which uses a wrong checksum calculation */
            if( (m_tRxCurrent.tPacket.tHeader.usChecksum != CalcCrc16((uint8_t*)m_tRxCurrent.tPacket.pbData,
                                                                      m_tRxCurrent.tPacket.tHeader.ulLength)) &&
                (m_tRxCurrent.tPacket.tHeader.usChecksum != CalcCrcNetIC((uint8_t*)m_tRxCurrent.tPacket.pbData,
                                                                      m_tRxCurrent.tPacket.tHeader.ulLength)) )
            {
              // Send negative acknowledge 
              SendAcknowledge(&m_tRxCurrent.tPacket.tHeader, HIL_TSTATE_CHECKSUM_ERROR);
              fDispatchPacket = false;
            }
          }
          m_tRxCurrent.eRxState = ePACKET_COMPLETE;
        }
      }
      break;

      case ePACKET_COMPLETE:  // State packet complete, 
      
      // Handle, receiving data packet               
      if(fDispatchPacket)
      {
        // Lock Receive part
        EnterCriticalSection(&m_tcsRxQueue);
        
        // Alocate memory for packet information        
        HIL_TRANSPORT_PACKET* ptTransPkt = AllocateTransportFrame(m_tRxCurrent.tPacket.tHeader.ulLength);

        _ASSERT(ptTransPkt);
        
        // Copy Header 
        ptTransPkt->tHeader = m_tRxCurrent.tPacket.tHeader;

        // Copy packet data 
        memcpy(ptTransPkt->pbData, m_tRxCurrent.tPacket.pbData, ptTransPkt->tHeader.ulLength);

        // Clean receiver packet data
        delete [] m_tRxCurrent.tPacket.pbData;

        /* Enqueue and signal thread */
        m_cRxQueue.push_back(ptTransPkt);

        // UnLock Receive part
        LeaveCriticalSection(&m_tcsRxQueue);

        ReleaseSemaphore(m_hRxSemaphore, 1, NULL);

      } else
      { // Clean receiver packet
        if(m_tRxCurrent.tPacket.pbData != NULL)
        {
          delete [] m_tRxCurrent.tPacket.pbData;
          m_tRxCurrent.tPacket.pbData = NULL;
        }
      }

      // Prepare for new packet 
      m_tRxCurrent.eRxState                 = eWAIT_FOR_COOKIE;
      m_tRxCurrent.ulDataOffset             = 0;
      m_tRxCurrent.ulHeaderOffset           = 0;
      m_tRxCurrent.tPacket.tHeader.ulCookie = 0;
      m_tRxCurrent.tPacket.pbData           = NULL;
      break;

    default:
      _ASSERT(false);
      break;
    }
  }

  // Remember last reception time, for timeout monitoring 
  m_ulLastRxTime = GetTickCount();

  LeaveCriticalSection(&m_tcsRxLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Pending packet queue handler
///   \param  ptPacket  Transport packet to be seached in the pending queue
///   \return true on success
///////////////////////////////////////////////////////////////////////////////////////////
bool CTransportLayer::HandlePendingTransfers(PHIL_TRANSPORT_PACKET ptPacket)
{
  bool fRet = false;

  /* Check if this packet belongs to a transfer */
  EnterCriticalSection(&m_tcsTransferLock);

  for(size_t uiIdx = 0; uiIdx < m_cvPendingTransfers.size(); ++uiIdx)
  {
    PHIL_TRANSPORT_PACKET ptTransfer = m_cvPendingTransfers[uiIdx]->ptSendPacket;
    
    if( (ptTransfer->tHeader.bChannel      == ptPacket->tHeader.bChannel)    &&
        (ptTransfer->tHeader.bDevice       == ptPacket->tHeader.bDevice)     &&
        (ptTransfer->tHeader.usDataType    == ptPacket->tHeader.usDataType)    )
    {
      if ( (ptTransfer->tHeader.bSequenceNr == ptPacket->tHeader.bSequenceNr) ||
           (m_tServerInfo.fIgnoreSequenceNr)                                     )
      {
        /* We've found our transfer */
        m_cvPendingTransfers[uiIdx]->tRecvPacket = *ptPacket;
        ::SetEvent(m_cvPendingTransfers[uiIdx]->hWaitEvent);
        fRet = true;
        break;
      }
    }
  }

  LeaveCriticalSection(&m_tcsTransferLock);

  return fRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Query Server Features, call the device for specific information 
///   \param  ulTimeout   Timeout for the QueryServerFeature call
///   \return CIFX_NO_ERROR, on succes
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CTransportLayer::QueryServerFeatures(uint32_t ulTimeout)
{

  HIL_TRANSPORT_PACKET tSendPacket = {0};
  HIL_TRANSPORT_PACKET tRecvPacket = {0};

  GeneratePacketHeader(&tSendPacket.tHeader);
  
  tSendPacket.tHeader.usDataType  = HIL_TRANSPORT_TYPE_QUERYSERVER;
  tSendPacket.tHeader.bChannel    = 0;
  tSendPacket.tHeader.bDevice     = 0;

  int32_t lRet = TransferPacket( &tSendPacket, 
                              &tRecvPacket, 
                              ulTimeout);
                            
  if(CIFX_NO_ERROR == lRet)
  {
    PHIL_TRANSPORT_ADMIN_QUERYSERVER_DATA_T ptServerInfo = (PHIL_TRANSPORT_ADMIN_QUERYSERVER_DATA_T)tRecvPacket.pbData;

    if(tRecvPacket.tHeader.ulLength < sizeof(*ptServerInfo))
    {
      lRet = CIFX_TRANSPORT_DATA_TOO_SHORT;
    } else
    {
      TRACE("%s: QueryServerFeatures response",
            m_pcInterface->GetInterfaceName().c_str());  
      TRACE("---------------------------------");
      TRACE("netX Marshaller V%d.%d.%d.%d", ptServerInfo->ulVersionMajor, 
                                            ptServerInfo->ulVersionMinor, 
                                            ptServerInfo->ulVersionBuild, 
                                            ptServerInfo->ulVersionRevision);
      TRACE("Server name: %s", ptServerInfo->szServerName);
      TRACE("Parallel services: %d", ptServerInfo->ulParallelServices);
      TRACE("Buffer size: %d", ptServerInfo->ulBufferSize);
      TRACE("Features: 0x%08x", ptServerInfo->ulFeatures);
      TRACE("Structure version: %d", ptServerInfo->ulStructVersion);
      TRACE("Supported Datatypes:");
      // Clear supported features list
      m_tServerInfo.ulFeatures         = ptServerInfo->ulFeatures;
      m_tServerInfo.ulBufferSize       = ptServerInfo->ulBufferSize;
      m_tServerInfo.ulParallelServices = ptServerInfo->ulParallelServices;
      m_tServerInfo.cvSupportedDataTypes.clear();
      m_tServerInfo.fIgnoreSequenceNr  = false;

      // Insert supported data types of interface partner
      for(size_t iIdx = 0; iIdx < ptServerInfo->ulDatatypeCnt; ++iIdx)
      {
        if(ptServerInfo->ausDataTypes[iIdx] == HIL_TRANSPORT_TYPE_KEEP_ALIVE)
        {
#ifdef _DEBUG
          TRACE("ATTENTION: KeepAlive is DEACTIVATED for DEBUG");
#else
          /* Device supports keep alive */
          m_eKeepAliveState = eKEEP_ALIVE_INITIALIZATION;

          /* Query keep alive timeout from connector settings */
          m_ulKeepAliveTimeout = m_pcInterface->GetKeepAliveTimeout();

          HandleKeepAlive();
#endif
        }

        TRACE("* 0x%04x", ptServerInfo->ausDataTypes[iIdx]);
        m_tServerInfo.cvSupportedDataTypes.push_back(ptServerInfo->ausDataTypes[iIdx]);

      }
      TRACE("---------------------------------");    
    }

    delete [] tRecvPacket.pbData;
  }

  if(CIFX_NO_ERROR != lRet)
  {
    m_tServerInfo.ulBufferSize       = sizeof(CIFX_PACKET);
    m_tServerInfo.ulParallelServices = 1;
    m_tServerInfo.cvSupportedDataTypes.clear();
    m_tServerInfo.fIgnoreSequenceNr  = true;
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Query device information, call the device for specific information
///   \param  usDataType      Data Type 
///   \param  ulOptions       Options of the Query Information call
///   \param  bDevice         Number of device
///   \param  bChannel        Number of channel
///   \param  ulBufferLen     Buffer size
///   \param  pvBuffer        Buffer to store information
///   \param  ulTimeout       Time out for the function call on device side 
///   \return CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CTransportLayer::QueryDeviceInformation(  uint16_t          usDataType,
                                               uint32_t           ulOptions, 
                                               BYTE                    bDevice, 
                                               BYTE                    bChannel,
                                               uint32_t           ulBufferLen,
                                               void*                   pvBuffer,
                                               uint32_t           ulTimeout)
{
  int32_t                            lRet              = CIFX_NO_ERROR;
  PHIL_TRANSPORT_PACKET           ptSendPkt         = {0};
  PHIL_TRANSPORT_PACKET           ptRecvPkt         = {0};
  uint32_t                   ulRequestSize     = 0;
  uint32_t                   ulSendDataSize    = sizeof(HIL_TRANSPORT_ADMIN_QUERYDEVICE_REQ_DATA_T);

  // Calculate the request size
  ulRequestSize = sizeof(ptSendPkt->tHeader)  + 
                  sizeof(ptSendPkt->pbData)   + 
                  ulSendDataSize;

  // Create a send buffer
  ptSendPkt = reinterpret_cast<PHIL_TRANSPORT_PACKET>(new uint8_t[ulRequestSize]);
  memset(ptSendPkt, 0, ulRequestSize);
  // Setup pointer to the content buffer
  uint8_t* pbData   = reinterpret_cast<uint8_t*>(ptSendPkt + 1);
  ptSendPkt->pbData       = pbData;
  size_t ulOffset         = 0;

  // Create information request
  memcpy(&pbData[ulOffset], &usDataType, sizeof(uint16_t));
  ulOffset += ( sizeof(usDataType) + sizeof(uint16_t));

  memcpy(&pbData[ulOffset], &ulOptions, sizeof(ulOptions));
  ulOffset += sizeof(ulOptions);
 
  GeneratePacketHeader(&ptSendPkt->tHeader);
  
  // Set device and channel number 
  ptSendPkt->tHeader.ulLength   = ulSendDataSize;
  ptSendPkt->tHeader.bChannel   = bChannel;
  ptSendPkt->tHeader.bDevice    = bDevice;
  ptSendPkt->tHeader.usDataType = HIL_TRANSPORT_TYPE_QUERYDEVICE;

  // Send Query device information request 
  if(CIFX_NO_ERROR == (lRet = TransferPacket( ptSendPkt, 
                                              ptRecvPkt,
                                              ulTimeout)))
  {
    // Handle received packet according to the given packet option
    switch(ulOptions)
    {
      case HIL_TRANSPORT_QUERYDEVICE_OPT_DEVICECNT:
      {
        PHIL_TRANSPORT_ADMIN_QUERYDEVICECNT_CNF_DATA_T ptInfo = (PHIL_TRANSPORT_ADMIN_QUERYDEVICECNT_CNF_DATA_T)ptRecvPkt->pbData;

        if(ptRecvPkt->tHeader.ulLength < sizeof(*ptInfo))
        {
          lRet = CIFX_TRANSPORT_DATA_TOO_SHORT;

        } else if(HIL_ADMIN_S_OK != ptInfo->ulError)
        {
          lRet = ptInfo->ulError;

        } else if (ulBufferLen < sizeof(BYTE))
        {
          lRet = CIFX_BUFFER_TOO_SHORT;

        } else
        {
          uint32_t* pulDeviceCnt = (uint32_t*)pvBuffer;
          *pulDeviceCnt      = ptInfo->ulDeviceCnt;
        }
      }
      break;

      case HIL_TRANSPORT_QUERYDEVICE_OPT_CHANNELCNT:
      {
        PHIL_TRANSPORT_ADMIN_QUERYCHANNELCNT_CNF_DATA_T ptInfo = (PHIL_TRANSPORT_ADMIN_QUERYCHANNELCNT_CNF_DATA_T)ptRecvPkt->pbData;

        if(ptRecvPkt->tHeader.ulLength < sizeof(*ptInfo))
        {
          lRet = CIFX_TRANSPORT_DATA_TOO_SHORT;

        } else if(HIL_ADMIN_S_OK != ptInfo->ulError)
        {
          lRet = CIFX_TRANSPORT_DATA_TOO_SHORT;

        } else if (ulBufferLen < sizeof(BYTE))
        {
          lRet = CIFX_BUFFER_TOO_SHORT;

        } else
        {
          uint32_t* pulChannelCnt = (uint32_t*)pvBuffer;
          *pulChannelCnt       = ptInfo->ulChannelCnt;
        }
      }
      break;

      case HIL_TRANSPORT_QUERYDEVICE_OPT_DEVICEINFO:
      {
        PHIL_TRANSPORT_ADMIN_QUERYDEVICEINFO_CNF_DATA_T ptInfo = (PHIL_TRANSPORT_ADMIN_QUERYDEVICEINFO_CNF_DATA_T)ptRecvPkt->pbData;

        if(ptRecvPkt->tHeader.ulLength < sizeof(*ptInfo))
        {
          lRet = CIFX_TRANSPORT_DATA_TOO_SHORT;

        } else if(HIL_ADMIN_S_OK != ptInfo->ulError)
        {
          lRet = CIFX_TRANSPORT_DATA_TOO_SHORT;

        } else if (ulBufferLen < sizeof(BOARD_INFORMATION))
        {
          lRet = CIFX_BUFFER_TOO_SHORT;

        } else
        {
          BOARD_INFORMATION* ptBoardInfo = (BOARD_INFORMATION*)pvBuffer;
          
          ptBoardInfo->tSystemInfo.ulDeviceNumber     = ptInfo->ulDeviceNr;
          ptBoardInfo->tSystemInfo.ulSerialNumber     = ptInfo->ulSerialNr;
          ptBoardInfo->tSystemInfo.usManufacturer     = ptInfo->usMfgNr;
          ptBoardInfo->tSystemInfo.usProductionDate   = ptInfo->usProdDate;
          ptBoardInfo->tSystemInfo.ulLicenseFlags1    = ptInfo->ulLicenseFlag1;
          ptBoardInfo->tSystemInfo.ulLicenseFlags2    = ptInfo->ulLicenseFlag2;
          ptBoardInfo->tSystemInfo.usNetxLicenseID    = ptInfo->usLicenseID;
          ptBoardInfo->tSystemInfo.usNetxLicenseFlags = ptInfo->usLicenseFlags;
          ptBoardInfo->tSystemInfo.usDeviceClass      = ptInfo->usDeviceClass;
          ptBoardInfo->tSystemInfo.bHwRevision        = ptInfo->bHwRevision;
          ptBoardInfo->tSystemInfo.bHwCompatibility   = ptInfo->bHwCompatibility;
        }
      }
      break;

      case HIL_TRANSPORT_QUERYDEVICE_OPT_CHANNELINFO:
      {
        PHIL_TRANSPORT_ADMIN_QUERYCHANNELINFO_CNF_DATA_T ptInfo = (PHIL_TRANSPORT_ADMIN_QUERYCHANNELINFO_CNF_DATA_T)ptRecvPkt->pbData;

        if(ptRecvPkt->tHeader.ulLength < sizeof(*ptInfo))
        {
          lRet = CIFX_TRANSPORT_DATA_TOO_SHORT;

        } else if(HIL_ADMIN_S_OK != ptInfo->ulError)
        {
          lRet = CIFX_TRANSPORT_DATA_TOO_SHORT;

        } else if (ulBufferLen < sizeof(CHANNEL_INFORMATION))
        {
          lRet = CIFX_BUFFER_TOO_SHORT;

        } else
        {
          CHANNEL_INFORMATION* ptChannelInfo = (CHANNEL_INFORMATION*)pvBuffer;
          strncpy((char*)ptChannelInfo->abFWName, (char*)ptInfo->szFWName, 63);
          ptChannelInfo->usFWMajor    = ptInfo->usFWMajor;
          ptChannelInfo->usFWMinor    = ptInfo->usFWMinor;
          ptChannelInfo->usFWBuild    = ptInfo->usFWBuild;
          ptChannelInfo->usFWRevision = ptInfo->usFWRev;
          ptChannelInfo->usFWYear     = ptInfo->usYear;
          ptChannelInfo->bFWMonth     = ptInfo->bMonth;
          ptChannelInfo->bFWDay       = ptInfo->bDay;
        }
      }
      break;

      default:
        // Invalid Device information option
        lRet = CIFX_INVALID_PARAMETER;
      break;
    }
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Send a specific packet (use the Hilscher Transport Header to identify packet)
///  \param   ptPacket                Pointer of the specific packet 
///  \param   pcLayer                 Pointer of DataLayer object( if NULL no layer object is use)
///  \param   ulTimeout               Time out of the send call
///  \param   fIgnoreActiveServiceCnt Packet will be send even if we exceed maximum parallel services
///  \return  CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CTransportLayer::SendPacket( PHIL_TRANSPORT_PACKET   ptPacket,
                                  CDataLayer*             pcLayer,
                                  uint32_t           ulTimeout,
                                  bool                    fIgnoreActiveServiceCnt /*  = false */)
{
  int32_t            lRet      = CIFX_NO_ERROR;
  INSERV_ENTRY_T  tInServ   = {0};
  uint8_t   bSeqNr    = ptPacket->tHeader.bSequenceNr;
  uint8_t   bAckState = 0;

  /* Are we communicating with a firmware with old transport protocol implementation ? */
  if (m_tServerInfo.fIgnoreSequenceNr)
  {
    /* Old netIC Firmware might be uses a wrong checksum calculation,
       so we don't calculate the checksum */
    ptPacket->tHeader.usChecksum = 0x0000;
  } else
  {
    /* Calculate packet checksum */
    ptPacket->tHeader.usChecksum = CalcCrc16((uint8_t*)ptPacket->pbData, 
                                             ptPacket->tHeader.ulLength);
  }

  // Allocate memory for ptPacket as well as save data
  HIL_TRANSPORT_PACKET* ptTransPkt = AllocateTransportFrame(ptPacket->tHeader.ulLength);
  
  _ASSERT(ptTransPkt);

  ptTransPkt->tHeader = ptPacket->tHeader;
  memcpy(ptTransPkt->pbData, ptPacket->pbData, ptPacket->tHeader.ulLength);

  tInServ.tPktHeader  = ptPacket->tHeader;
  tInServ.pbAckState  = &bAckState;

  if(pcLayer == NULL)
  {
    tInServ.hWaitAckEvent   = ::CreateEvent(NULL, FALSE, FALSE, NULL);

  } else
  {
    tInServ.pcLayer = pcLayer;
  }

  /* Insert into Send queue and IN-Service Queue */
  EnterCriticalSection(&m_tcsInService);

  // Insert Inservice map entry
  m_cmInService[bSeqNr] = tInServ;

  LeaveCriticalSection(&m_tcsInService);

  EnterCriticalSection(&m_tcsTxQueue);
  
  m_cTxQueue.push_back(ptTransPkt);

  /* NOTE: we added our request to in-service, so we need must signal semaphore
     if size of map equals to ulParallelServices */
  AddParallelService(CIFX_TO_1ST_PACKET, fIgnoreActiveServiceCnt);


  LeaveCriticalSection(&m_tcsTxQueue);

  /* Send OK */
  if(pcLayer == NULL)
  {
    /* No callback wanted, so wait for event here */
    DWORD dwWaitRes = ::WaitForSingleObject(tInServ.hWaitAckEvent, ulTimeout + m_ulSendTimeout);
    
    switch(dwWaitRes)
    {
      case WAIT_OBJECT_0:
        lRet = RetFromState(bAckState);
      break;

      case WAIT_TIMEOUT:
        { 
          /* Insert into Send queue and IN-Service Queue */
          EnterCriticalSection(&m_tcsInService);

          HIL_INSERV_MAP::iterator iter = m_cmInService.find(bSeqNr);
          
          if(iter != m_cmInService.end())
          {
            m_cmInService.erase(iter);
          }

          LeaveCriticalSection(&m_tcsInService);

          EnterCriticalSection(&m_tcsTxQueue);
          if(m_cTxQueue.size() > 0)
            m_cTxQueue.pop_front();
          LeaveCriticalSection(&m_tcsTxQueue);
       
          lRet = CIFX_DEV_PUT_TIMEOUT;
        }
        break;

      default:
        _ASSERT(false);
        break;
    }
  }

  if(tInServ.hWaitAckEvent != NULL)
  {
    ::CloseHandle(tInServ.hWaitAckEvent);
  }
  
  if (CIFX_NO_ERROR != lRet)
  {
    /* Error returned in acknowledge packet, so we need to remove the active service */
    RemoveParallelService();
    if (m_cTxQueue.size() > 0)     
      AddParallelService(CIFX_TO_1ST_PACKET);
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Transport packet, set also parameter of the transport header  
///   \param    usDataType      Data type of the packet
///   \param    pbSendData      Reference of send data 
///   \param    ulSendDataLen   Length of the send data 
///   \param    pbRecvData      Reference where the received data should be after recv
///   \param    ulRecvDataLen   Length of the Recv data
///   \param    pcEndpoint      Reference of device
///   \param    bChannel        Channel number for the transport packet header
///   \param    ulTimeout       Time out value for the transmission
///   \return   CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CTransportLayer::TransferPacket( WORD              usDataType,
                                      uint8_t*          pbSendData,
                                      uint32_t          ulSendDataLen,
                                      uint8_t*&         pbRecvData,
                                      uint32_t&         ulRecvDataLen,
                                      CEndpoint*        pcEndpoint,
                                      uint8_t           bChannel,
                                      uint32_t          ulTimeout)
{
  // Check lock of transfer packet, to be sure that no data are changed before they are called to send
  HIL_TRANSPORT_PACKET tSendHilPacket = {0};
  HIL_TRANSPORT_PACKET tRecvHilPacket = {0};
                                                                       
  tSendHilPacket.pbData             = pbSendData;
  tSendHilPacket.tHeader.usDataType = usDataType;
  tSendHilPacket.tHeader.ulLength   = ulSendDataLen;
  tSendHilPacket.tHeader.bDevice    = pcEndpoint->m_bDevice;
  tSendHilPacket.tHeader.bChannel   = bChannel;

  GeneratePacketHeader(&tSendHilPacket.tHeader);

  int32_t lRet = TransferPacket( &tSendHilPacket,
                              &tRecvHilPacket, 
                              ulTimeout);

  if(CIFX_NO_ERROR == lRet)
  {
    pbRecvData    = tRecvHilPacket.pbData;
    ulRecvDataLen = tRecvHilPacket.tHeader.ulLength;
  }
  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Transfer packet, send and receive packet, syncron call of transfer 
///   \param  ptSendPacket            pointer to the send packet 
///   \param  ptRecvPacket            pointer to the received packet 
///   \param  ulTimeout               time out for the receive data 
///   \param  fIgnoreActiveServiceCnt Packet will be send even if we exceed maximum parallel services
///   \return CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CTransportLayer::TransferPacket( PHIL_TRANSPORT_PACKET   ptSendPacket, 
                                      PHIL_TRANSPORT_PACKET   ptRecvPacket,
                                      uint32_t           ulTimeout,
                                      bool                    fIgnoreActiveServiceCnt /*  = false */)
{
  // Check incomming parameters 
  if( (NULL == ptSendPacket) ||
      (NULL == ptRecvPacket) )
    return CIFX_INVALID_POINTER;

  int32_t lRet = CIFX_TRANSPORT_CONNECT;

  if (m_pcInterface->GetState() == eINTERFACE_STATE_RUNNING)
  {
    bool                fRunning        = true;

    /* Enqueue Transfer to wait for data */
    TRANSFER_INFO_T tTransfer = {0};
    tTransfer.hWaitEvent      = ::CreateEvent(NULL, FALSE, FALSE, NULL);  // Event for the receive data 
    tTransfer.ptSendPacket    = ptSendPacket;
    
    // Save transfer information on device pending queue    
    EnterCriticalSection(&m_tcsTransferLock);

    m_cvPendingTransfers.push_back(&tTransfer);

    LeaveCriticalSection(&m_tcsTransferLock);
    
    // Send data also wait for acknowledge
    if(CIFX_NO_ERROR == (lRet = SendPacket( ptSendPacket,
                                            NULL,
                                            ulTimeout,
                                            fIgnoreActiveServiceCnt)))
    {
      HIL_INSERV_MAP::iterator iter;

      while(fRunning)
      {
        // Wait for answer 
        DWORD dwWaitRes = ::WaitForSingleObject(tTransfer.hWaitEvent, m_ulSendTimeout + ulTimeout);
        switch(dwWaitRes)
        {
          case WAIT_OBJECT_0: // Receive Data
          {
            // Store header 
            if(tTransfer.fAborted)
            {
              /* This transfer has been aborted (e.g. KeepAlive Timeout or detach) */
              lRet = CIFX_TRANSPORT_ABORTED;
            } else
            {
              memcpy( &ptRecvPacket->tHeader, 
                      &tTransfer.tRecvPacket.tHeader,
                      sizeof(ptRecvPacket->tHeader));

              // Store data 
              ptRecvPacket->pbData = tTransfer.tRecvPacket.pbData;            
             }
            fRunning = false;
          }
          break;

          case WAIT_TIMEOUT:
          {
            lRet = CIFX_TRANSPORT_RECV_TIMEOUT;
            fRunning = false;
          }
          break;

          default:
            _ASSERT(false);
          break;
        }
      }
    }

    if(tTransfer.hWaitEvent != NULL)
      ::CloseHandle(tTransfer.hWaitEvent);
    
    /* Remove packet from transfer queue */
    EnterCriticalSection(&m_tcsTransferLock);

    for(size_t uiIdx = 0; uiIdx < m_cvPendingTransfers.size(); ++uiIdx)
    {
      if(m_cvPendingTransfers[uiIdx] == &tTransfer)
      {
        m_cvPendingTransfers.erase(m_cvPendingTransfers.begin() + uiIdx);
        break;
      }
    }
    LeaveCriticalSection(&m_tcsTransferLock);

    /* Check if we have to signal more transfers */
    if(lRet == CIFX_TRANSPORT_RECV_TIMEOUT)
    {
      /* Transfer timed out, so we need to remove the active service */
      RemoveParallelService();
    }

    if (m_cTxQueue.size() > 0)     
      AddParallelService(CIFX_TO_1ST_PACKET);
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Send acknowledge via connector, 
///   \param  ptHeader  HIL packet header
///   \param  bState    Acknowledeg state
///////////////////////////////////////////////////////////////////////////////////////////
void CTransportLayer::SendAcknowledge(PHIL_TRANSPORT_HEADER ptHeader, uint8_t bState)
{
  if(NULL != ptHeader)
  {
    m_tACKPacket.bChannel       = ptHeader->bChannel;
    m_tACKPacket.bDevice        = ptHeader->bDevice;
    m_tACKPacket.bSequenceNr    = ptHeader->bSequenceNr;
    m_tACKPacket.bState         = bState;
    m_tACKPacket.usTransaction  = ptHeader->usTransaction;
    // Send acknowlegde 
    if (CIFX_NO_ERROR != m_pcInterface->Send((uint8_t*)&m_tACKPacket, sizeof(m_tACKPacket)))
    {
      /* Tell Devicehandler we want to reconnect */
      m_pcDeviceHandler->ScheduleInterfaceMonitor(m_pcInterface->GetInterfaceName(), CDeviceHandler::eRECONNECT);
    }

  }else
  {
    _ASSERT(ptHeader);
  }
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Get the send mailbox state for non API functions
///  \param ulDevice          Device number
///  \param &ulSendMBXState   State of the send mailbox
///  \return true on success
///////////////////////////////////////////////////////////////////////////////////////////
bool CTransportLayer::GetSendMbxState(uint32_t ulDevice, uint32_t& ulSendMBXState)
{
  UNREFERENCED_PARAMETER(ulDevice);
  ulSendMBXState = (uint32_t)(m_tServerInfo.ulParallelServices - m_cvActiveServices.size());

  return true;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Increment the number of parallel services
/// \param ulTimeout               Timeout of parallel service
/// \param fIgnoreActiveServiceCnt Ignor maximum allowed parallel services
/// \return TRUE if service is successfully added
///////////////////////////////////////////////////////////////////////////////////////////
bool CTransportLayer::AddParallelService(uint32_t ulTimeout, bool fIgnoreActiveServiceCnt /* = false */)
{
  bool fRet = false;

  EnterCriticalSection(&m_tcsActiveServices);

  if( (m_cvActiveServices.size() < m_tServerInfo.ulParallelServices) || 
      fIgnoreActiveServiceCnt )
  {
    DWORD dwParallelServiceTimeout  = GetTickCount() + ulTimeout;
    m_cvActiveServices.push_back(dwParallelServiceTimeout);
    m_cvActiveServices.sort();

    /* Only signal thread to send data, if parallel services
       is not exceeded. When the next RX packet arrives, or 
       a timeout occurs, the Semaphore will be signalled automatically by
       RxHandler */
    ReleaseSemaphore(m_hTxSemaphore, 1, NULL);

    /* TODO: Who releases semaphore in case maximum parallel services are
             reached and Asynchronous mode (rcX Packet is used)? (Timeout) */
    fRet = true;
  }
  
  LeaveCriticalSection(&m_tcsActiveServices);  
  
  return fRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Decrement the number of parallel services
///////////////////////////////////////////////////////////////////////////////////////////
void CTransportLayer::RemoveParallelService(void)
{
  EnterCriticalSection(&m_tcsActiveServices);

  if (m_cvActiveServices.size() > 0)
    m_cvActiveServices.pop_front();
  
  LeaveCriticalSection(&m_tcsActiveServices);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Check timeouts of parallel services
///////////////////////////////////////////////////////////////////////////////////////////
void CTransportLayer::HandleParallelServiceTimeout(void)
{
  if (m_cvActiveServices.size() == 0)
    return;

  uint32_t ulServiceTimeout = 0;
  uint32_t ulCurTime        = GetTickCount();

  EnterCriticalSection(&m_tcsActiveServices);
  
  while ((0 < m_cvActiveServices.size()) && (m_cvActiveServices.front() <= ulCurTime))
  {
    m_cvActiveServices.pop_front();
    ulServiceTimeout++;
  }

  LeaveCriticalSection(&m_tcsActiveServices);

  for (uint32_t ulIdx = 0; ulIdx<min(ulServiceTimeout, m_cTxQueue.size()); ulIdx++)
  {
    AddParallelService(CIFX_TO_1ST_PACKET);
  }
}


///////////////////////////////////////////////////////////////////////////////////////////
/// Handle an incoming keep-alive answer
///////////////////////////////////////////////////////////////////////////////////////////
void CTransportLayer::HandleKeepAlive(void)
{
  switch(m_eKeepAliveState)
  {
    case eKEEP_ALIVE_UNSUPPORTED:
      /* Nothing to do, as the device does not support keep alive */
    break;

    case eKEEP_ALIVE_INITIALIZATION:
    {
      HIL_TRANSPORT_PACKET tRecvPacket = {0};

      m_ulKeepAliveIdent = 0;
      GeneratePacketHeader(&m_tKeepAlivePacket.tHeader);

      if(CIFX_NO_ERROR != TransferPacket(&m_tKeepAlivePacket, &tRecvPacket, TRANSPORT_TO_TRANSFER))
      {
        /* We were unable to get a keep valid alive sequence number,
          so we assume the device does not support keep alive correctly */
        m_eKeepAliveState = eKEEP_ALIVE_UNSUPPORTED;

      } else
      {
        PHIL_TRANSPORT_KEEPALIVE_DATA_T ptKeepAliveData = (PHIL_TRANSPORT_KEEPALIVE_DATA_T)tRecvPacket.pbData;

        /* Store COMId for verification */
        m_ulKeepAliveIdent = ptKeepAliveData->ulComID;
        m_eKeepAliveState  = eKEEP_ALIVE_ACTIVE;
        
        m_ulLastKeepAlive = GetTickCount();

        delete [] tRecvPacket.pbData;
      }
    }
    break;

    case eKEEP_ALIVE_ACTIVE:
    {
      uint32_t ulTimeDiff = abs((int32_t)GetTickCount() - (int32_t)m_ulLastKeepAlive);

      if(ulTimeDiff > m_ulKeepAliveTimeout)
      {
        HIL_TRANSPORT_PACKET tRecvPacket = {0};

        GeneratePacketHeader(&m_tKeepAlivePacket.tHeader);

        if(CIFX_NO_ERROR != TransferPacket(&m_tKeepAlivePacket, &tRecvPacket, m_ulKeepAliveTimeout, true))
        {
          
          TRACE("Keep alive failed on interface %s!", m_pcInterface->GetInterfaceName().c_str());

          /* Assume Device disconnect, as it does not answer */
          m_eKeepAliveState = eKEEP_ALIVE_TIMEOUT;

          /* Notify data layers of disconnect */
          m_pcDeviceHandler->ScheduleInterfaceMonitor(m_pcInterface->GetInterfaceName(), CDeviceHandler::eRECONNECT);
        } else
        {
          /* COMId should be verified by device, so we can be sure, that the device 
            has answered our keep alive */
          delete [] tRecvPacket.pbData;
          
          m_ulLastKeepAlive = GetTickCount();
        }
      }
    }
    break;

    case eKEEP_ALIVE_TIMEOUT:
    break;

    default:
    break;
  }
}

DWORD CTransportLayer::KeepAliveThread(void* pvParam)
{
  CTransportLayer* pcLayer  = reinterpret_cast<CTransportLayer*>(pvParam);
  HANDLE           ahWait[] = {pcLayer->m_hKeepAliveStop, pcLayer->m_hKeepAliveEvent};
  bool             fRunning = true;

  do
  {
    DWORD dwWait = ::WaitForMultipleObjects(sizeof(ahWait) / sizeof(ahWait[0]),
                                            ahWait,
                                            FALSE,
                                            INFINITE);

    switch(dwWait)
    {
    default:
    case WAIT_OBJECT_0:
      fRunning = false;
      break;

    case WAIT_OBJECT_0 + 1:
      pcLayer->HandleKeepAlive();
      break;
    }


  } while(fRunning);

  return 0;
}
